﻿using System;
using System.Collections.Generic;
using System.IdentityModel.Tokens.Jwt;
using System.Linq;
using System.Security.Claims;
using System.Xml.Linq;
using Vimt.JsonWebToken.Messages.Messages;
using Vimt.JsonWebToken.Services.Configuration;
using Vimt.JsonWebToken.Services.Rest;
using VRM.Integration.Servicebus.Core;
using Logger = VRM.Integration.Servicebus.Core.Logger;

namespace Vimt.JsonWebToken.Processors
{
    /// <summary>
    /// VIMT JWT Encrypt Processor.
    /// </summary>
    public class VimtJwtEncryptProcessor
    {
        /// <summary>
        /// Execute VIMT handler.
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public IMessageBase Execute(VimtJwtEncryptTokenRequest request)
        {
            try
            {
                if (request == null) throw new Exception("VimtJwtEncryptTokenRequest message is null");
                if (string.IsNullOrEmpty(request.SamlToken)) throw new Exception("SAML token is null or empty");

                var xdoc = XDocument.Parse(request.SamlToken);

                var firstNameAttribute = xdoc.Descendants().FirstOrDefault(x => (string)x.Attribute("Name") == "urn:va:vrm:iam:firstname");
                if (firstNameAttribute == null) throw new Exception("First Name attribute in SAML token cannot be missing or null");
                var firstName = firstNameAttribute.Value;

                var lastNameAttribute = xdoc.Descendants().FirstOrDefault(x => (string)x.Attribute("Name") == "urn:va:vrm:iam:lastname");
                if (lastNameAttribute == null) throw new Exception("Last Name attribute in SAML token cannot be missing or null");
                var lastName = lastNameAttribute.Value;

                var secIdAttribute = xdoc.Descendants().FirstOrDefault(x => (string)x.Attribute("Name") == "urn:va:vrm:iam:secid");
                if (secIdAttribute == null) throw new Exception("SecID attribute in SAML token cannot be missing or null");
                var secId = secIdAttribute.Value;

                var subIdAttribute = xdoc.Descendants().FirstOrDefault(x => (string)x.Attribute("Name") == "urn:oasis:names:tc:xspa:1.0:subject:subject-id");
                if (subIdAttribute == null) throw new Exception("Subject ID attribute in SAML token cannot be missing or null");
                var subId = subIdAttribute.Value;

                var vistAttribute = xdoc.Descendants().FirstOrDefault(x => (string)x.Attribute("Name") == "urn:va:vrm:iam:vistaid");
                if (vistAttribute == null) throw new Exception("Vista ID attribute in SAML token cannot be missing or null");
                var vistaNodes = vistAttribute.Nodes().ToList();
                if (vistaNodes.Count < 1) throw new Exception("Vista ID attribute in SAML token has no Vista IDs");

                var vistas = new List<object>();

                foreach (var node in vistaNodes)
                {
                    var value = ((XElement)node).Value;
                    var split = value.Split('-');
                    vistas.Add(new { siteId = split[0], duz = split[1] });
                }

                var nbf = DateTime.UtcNow.AddSeconds(-1);
                var exp = DateTime.UtcNow.AddSeconds(120);
                var payload = new JwtPayload(JwtSecurityConfiguration.Current.Issuer, string.Empty, new List<Claim>(), nbf, exp)
                {
                    {"authenticated", JwtSecurityConfiguration.Current.Authenticated},
                    {"sub", subId},
                    {"lastName", lastName},
                    {"authenticationAuthority", JwtSecurityConfiguration.Current.AuthenticationAuthority},
                    {"idType", JwtSecurityConfiguration.Current.IdType},
                    {"vamf.auth.resources", new[] {"^.*(\\/)?patient[s]?((\\/.*$)|(\\?.*$)|($))", "^.*(\\/)?staff\\/1234567890\\/.*$"}},
                    {"vistaIds", vistas.ToArray()},
                    {"firstName", firstName},
                    {"attributes", new {secid = secId}},
                    {"vamf.auth.roles", new[] {JwtSecurityConfiguration.Current.VamfAuthRole}},
                    {"jti", Guid.NewGuid().ToString()},
                    {"loa", JwtSecurityConfiguration.Current.Loa}
                };

                var token = ServiceFactory.EncryptToken(payload).GetAwaiter().GetResult();

                return new VimtJwtEncryptTokenResponse { EncryptedJwtToken = token };
            }
            catch (Exception ex)
            {
                Logger.Instance.Error("There was an error creating the Json Web Token: Error", ex);
                return new VimtJwtEncryptTokenResponse { ExceptionOccured = true, ExceptionMessage = ex.Message };
            }
        }
    }
}